import {Box, Paper} from "@mui/material";
import type {MapFile, MapLayer} from "../_types/mapTypes";
import type {SpriteEntry} from "../_types/spriteTypes";

type Props = {
    map: MapFile;
    onCellClick?: (row: number, col: number) => void;
    onCellHover?: (pos: { row: number; col: number } | null) => void;
    allSprites: SpriteEntry[];
    selectedSprite?: SpriteEntry | null;
    hoverCell?: { row: number; col: number } | null;
};

function cellContentAllLayers(layers: MapLayer[], row: number, col: number, allSprites: SpriteEntry[]) {
    // On empile chaque sprite sur la même case si visible
    const visibleLayers = layers.filter(l => l.visible);
    return visibleLayers.map((layer, idx) => {
        const cell = layer.cells[row][col];
        if (!cell) return null;
        const sprite = allSprites.find(s => s.key === cell.sprite);
        if (!sprite) return null;
        return (
            <canvas
                key={layer.id}
                width={48}
                height={48}
                ref={ref => {
                    if (ref && sprite) {
                        const ctx = ref.getContext("2d")!;
                        ctx.clearRect(0, 0, 48, 48);
                        sprite.drawFn(ctx, 0, 0, {w: 48, h: 48});
                    }
                }}
                style={{
                    width: "100%",
                    height: "100%",
                    position: "absolute",
                    top: 0, left: 0,
                    zIndex: 10 + idx,
                    pointerEvents: "none",
                }}
            />
        );
    });
}

export default function MapEditor({
                                      map,
                                      onCellClick,
                                      onCellHover,
                                      allSprites,
                                      selectedSprite,
                                      hoverCell,
                                  }: Props) {
    const activeLayer = map.layers.find(l => l.visible) || map.layers[0];
    return (
        <Box
            sx={{
                flex: 1,
                display: "flex",
                alignItems: "center",
                justifyContent: "center",
                overflow: "auto",
                bgcolor: "#e9eef2",
            }}
        >
            <Paper
                elevation={3}
                sx={{
                    display: "inline-block",
                    p: 2,
                    borderRadius: 3,
                    bgcolor: "#fff",
                }}
            >
                <Box
                    sx={{
                        display: "grid",
                        gridTemplateColumns: `repeat(${map.width}, ${map.tileSize}px)`,
                        gridTemplateRows: `repeat(${map.height}, ${map.tileSize}px)`,
                        gap: 0,
                        border: `2px solid #bbb`,
                        boxShadow: "0 2px 8px #0001",
                        background: "#c9d4e0",
                        userSelect: "none",
                    }}
                >
                    {Array.from({length: map.height}).map((_, row) =>
                        Array.from({length: map.width}).map((_, col) => {
                            const cell = activeLayer.cells[row][col];
                            const isHover = hoverCell && hoverCell.row === row && hoverCell.col === col;
                            return (
                                <Box
                                    key={row + "-" + col}
                                    onClick={() => onCellClick?.(row, col)}
                                    onMouseEnter={() => onCellHover?.({row, col})}
                                    onMouseLeave={() => onCellHover?.(null)}
                                    sx={{
                                        width: map.tileSize,
                                        height: map.tileSize,
                                        border: `1px solid #bbb`,
                                        background: isHover
                                            ? "#d2e3fc"
                                            : cell
                                                ? "#fff"
                                                : "#f6fafd",
                                        position: "relative",
                                        cursor: "pointer",
                                        boxShadow: isHover ? "0 0 8px #90caf9" : undefined,
                                        transition: "background 0.12s, box-shadow 0.12s",
                                    }}
                                >
                                    {cellContentAllLayers(map.layers, row, col, allSprites)}
                                    {/* Feedback visuel du sprite à placer */}
                                    {isHover && selectedSprite && !cell && (
                                        <canvas
                                            width={48}
                                            height={48}
                                            ref={ref => {
                                                if (ref) {
                                                    const ctx = ref.getContext("2d")!;
                                                    ctx.clearRect(0, 0, 48, 48);
                                                    selectedSprite.drawFn(ctx, 0, 0, {w: 48, h: 48, opacity: 0.4});
                                                }
                                            }}
                                            style={{
                                                width: "100%",
                                                height: "100%",
                                                position: "absolute",
                                                left: 0,
                                                top: 0,
                                                pointerEvents: "none",
                                                opacity: 0.5,
                                            }}
                                        />
                                    )}
                                </Box>
                            );
                        })
                    )}
                </Box>
            </Paper>
        </Box>
    );
}
