import { app, shell, BrowserWindow, ipcMain } from 'electron'
import { join } from 'path'
import { electronApp, optimizer, is } from '@electron-toolkit/utils'
import icon from '../../resources/icon.png?asset'

import { Low } from 'lowdb'
import { JSONFile } from 'lowdb/node'
import path from 'path'

// Type definitions
type Product = { id: string; name: string; price: number; stock: number };
type Sale = {
  id: string;
  items: { id: string; name: string; qty: number; price: number }[];
  total: number;
  date: string
};
type Data = { products: Product[]; sales: Sale[] };

// LowDB setup (simple JSON file DB)
const dbFile = path.join(__dirname, '../../db/data.json');
const adapter = new JSONFile<Data>(dbFile);
const defaultData: Data = { products: [], sales: [] };
const db = new Low<Data>(adapter, defaultData);


(async () => {
  await db.read();
  db.data ||= { products: [], sales: [] };   // 👈 Ici !
  await db.write();
})();

// Handlers pour IPC

ipcMain.handle('get-products', async (): Promise<Product[]> => {
  await db.read();
  return db.data?.products || [];
});

ipcMain.handle('add-product', async (_e, prod): Promise<Product[]> => {
  await db.read();
  const p = { ...prod, id: Date.now().toString() };
  db.data.products.push(p);
  await db.write();
  return p;
});

ipcMain.handle('delete-product', async (_e, id): Promise<Product[]> => {
  await db.read();
  db.data.products = db.data.products.filter(p => p.id !== id);
  await db.write();
  return true;
});

ipcMain.handle('get-sales', async (): Promise<Sale[]> => {
  await db.read();
  return db.data?.sales || [];
});

ipcMain.handle('sell-cart', async (_e, items, printerIp): Promise<Product[]> => {
  await db.read();
  let total = 0;
  const products = db.data.products;
  const saleItems = items.map(({ id, qty }) => {
    const p = products.find(x => x.id === id);
    total += p.price * qty;
    p.stock -= qty;
    return { id, name: p.name, price: p.price, qty };
  });
  const sale = { id: Date.now().toString(), items: saleItems, total, date: new Date().toISOString() };
  db.data.sales.push(sale);
  await db.write();
  // Impression réseau ici (ou met en commentaire si tu ne veux pas tester tout de suite)
  //let url = "http://192.168.1.125:9000/cgi-bin/epos/service.cgi?devid=local_printer&timeout=60000";

  return { ok: true, sale };
});
ipcMain.handle('print-ticket', async (_event, ticketXml, printerUrl) => {
  try {
    const response = await fetch(printerUrl, {
      method: 'POST',
      headers: { 'Content-Type': 'text/xml' },
      body: ticketXml
    });
    const result = await response.text();
    // Tu peux logguer la réponse ici
    console.log('Impression, retour Epson/mock :', result);
    return { success: true, response: result };
  } catch (err) {
    console.error('Erreur impression Epson:', err);
    return { success: false, error: err.message };
  }
});
function createWindow(): void {
  // Create the browser window.
  const mainWindow = new BrowserWindow({
    width: 900,
    height: 670,
    show: false,
    autoHideMenuBar: true,
    ...(process.platform === 'linux' ? { icon } : {}),
    webPreferences: {
      preload: join(__dirname, '../preload/index.js'),
      sandbox: false
    }
  })

  mainWindow.on('ready-to-show', () => {
    mainWindow.show()
  })

  mainWindow.webContents.setWindowOpenHandler((details) => {
    shell.openExternal(details.url)
    return { action: 'deny' }
  })

  // HMR for renderer base on electron-vite cli.
  // Load the remote URL for development or the local html file for production.
  if (is.dev && process.env['ELECTRON_RENDERER_URL']) {
    mainWindow.loadURL(process.env['ELECTRON_RENDERER_URL'])
  } else {
    mainWindow.loadFile(join(__dirname, '../renderer/index.html'))
  }
}

// This method will be called when Electron has finished
// initialization and is ready to create browser windows.
// Some APIs can only be used after this event occurs.
app.whenReady().then(() => {
  // Set app user model id for windows
  electronApp.setAppUserModelId('com.electron')

  // Default open or close DevTools by F12 in development
  // and ignore CommandOrControl + R in production.
  // see https://github.com/alex8088/electron-toolkit/tree/master/packages/utils
  app.on('browser-window-created', (_, window) => {
    optimizer.watchWindowShortcuts(window)
  })

  // IPC test
  ipcMain.on('ping', () => console.log('pong'))

  createWindow()

  app.on('activate', function () {
    // On macOS it's common to re-create a window in the app when the
    // dock icon is clicked and there are no other windows open.
    if (BrowserWindow.getAllWindows().length === 0) createWindow()
  })
})

// Quit when all windows are closed, except on macOS. There, it's common
// for applications and their menu bar to stay active until the user quits
// explicitly with Cmd + Q.
app.on('window-all-closed', () => {
  if (process.platform !== 'darwin') {
    app.quit()
  }
})

// In this file you can include the rest of your app's specific main process
// code. You can also put them in separate files and require them here.
