import React, { useState } from "react";

// Produits de démo (tu peux charger depuis une DB plus tard)
const PRODUITS_DEMO = [
  { id: 1, name: "Baguette", price: 1.2, stock: 10 },
  { id: 2, name: "Pain au chocolat", price: 1.5, stock: 15 },
  { id: 3, name: "Croissant", price: 1.3, stock: 12 },
  { id: 4, name: "Café", price: 1.0, stock: 30 }
];

const USERS = [
  { id: 1, name: "Alice", role: "vendeur" },
  { id: 2, name: "Bob", role: "admin" }
];

// Génération du XML ticket Epson
function generateTicketXML(cart = [], user) {
  const total = cart.reduce((sum, item) => sum + item.price * item.qty, 0).toFixed(2);
  return `<?xml version="1.0" encoding="utf-8"?>
<epos-print>
  <text align="center">Ticket de caisse</text>
  <text>Vendeur: ${user?.name || "?"}</text>
  <feed />
  ${cart.map(item => `
    <text>${item.qty} x ${item.name} ... ${(item.price * item.qty).toFixed(2)} €</text>
  `).join("")}
  <feed />
  <text>Total : ${total} €</text>
  <feed />
  <text>Merci !</text>
</epos-print>`;
}

export default function App() {
  const [products, setProducts] = useState(PRODUITS_DEMO);
  const [cart, setCart] = useState([]);
  const [printerIp, setPrinterIp] = useState("192.168.1.25:9000");
  const [status, setStatus] = useState("");
  const [user, setUser] = useState(USERS[0]);
  const [sales, setSales] = useState([]);
  const [admin, setAdmin] = useState(false);

  // Ajouter au panier
  const addToCart = (prod) => {
    if (prod.stock <= 0) return setStatus("Plus de stock !");
    setCart(c =>
      c.some(item => item.id === prod.id)
        ? c.map(item =>
          item.id === prod.id ? { ...item, qty: item.qty + 1 } : item
        )
        : [...c, { ...prod, qty: 1 }]
    );
    setProducts(ps => ps.map(p =>
      p.id === prod.id ? { ...p, stock: p.stock - 1 } : p
    ));
    setStatus("");
  };

  // Retirer du panier
  const removeFromCart = (prodId) => {
    const item = cart.find(i => i.id === prodId);
    setCart(c => c.filter(i => i.id !== prodId));
    setProducts(ps =>
      ps.map(p =>
        p.id === prodId ? { ...p, stock: p.stock + (item?.qty || 0) } : p
      )
    );
  };

  // Impression du ticket + validation vente
  const handlePrint = async () => {
    if (!cart.length) return setStatus("Panier vide !");
    const ticketXml = generateTicketXML(cart, user);
    const printerUrl = `http://${printerIp}/cgi-bin/epos/service.cgi?devid=local_printer&timeout=60000`;
    try {
      setStatus("Impression en cours...");
      const result = await window.api.printTicket(ticketXml, printerUrl);
      if (result.success) {
        setStatus("Impression réussie !");
        setSales(s => [
          ...s,
          {
            id: Date.now(),
            date: new Date().toLocaleString(),
            cart: [...cart],
            total: cart.reduce((sum, item) => sum + item.price * item.qty, 0),
            user: user.name
          }
        ]);
        setCart([]);
      } else setStatus("Erreur impression : " + result.error);
    } catch (e) {
      setStatus("Erreur impression : " + e.message);
    }
  };

  // Ajout produit (admin)
  const addProduct = () => {
    const name = prompt("Nom du produit ?");
    const price = parseFloat(prompt("Prix ?"));
    const stock = parseInt(prompt("Stock initial ?"));
    if (name && !isNaN(price) && !isNaN(stock)) {
      setProducts(ps => [
        ...ps,
        { id: Date.now(), name, price, stock }
      ]);
    }
  };

  // Suppression produit (admin)
  const deleteProduct = (id) =>
    setProducts(ps => ps.filter(p => p.id !== id));

  // Sélection utilisateur (change droits admin)
  const handleUser = (e) => {
    const u = USERS.find(u => u.id === parseInt(e.target.value));
    setUser(u);
    setAdmin(u?.role === "admin");
  };

  return (
    <div style={{ padding: 20, fontFamily: "sans-serif", maxWidth: 650 }}>
      <h2>Caisse</h2>

      <div style={{ marginBottom: 16 }}>
        <label>
          Utilisateur :
          <select value={user.id} onChange={handleUser} style={{ marginLeft: 8 }}>
            {USERS.map(u => (
              <option key={u.id} value={u.id}>{u.name} ({u.role})</option>
            ))}
          </select>
        </label>
        <label style={{ marginLeft: 16 }}>
          IP:Port imprimante :
          <input
            value={printerIp}
            onChange={e => setPrinterIp(e.target.value)}
            style={{ marginLeft: 8, width: 180 }}
          />
        </label>
      </div>

      <h3>Produits</h3>
      <div style={{ display: "flex", gap: 10, flexWrap: "wrap" }}>
        {products.map(prod => (
          <button
            key={prod.id}
            onClick={() => addToCart(prod)}
            disabled={prod.stock <= 0}
            style={{
              background: prod.stock > 0 ? "#fff" : "#ccc",
              border: "1px solid #999",
              borderRadius: 8,
              padding: "8px 16px"
            }}
          >
            {prod.name} ({prod.price} €) | Stock : {prod.stock}
            {admin && (
              <span style={{ marginLeft: 8, color: "red", cursor: "pointer" }}
                    onClick={e => {
                      e.stopPropagation();
                      deleteProduct(prod.id);
                    }}>
                [Suppr]
              </span>
            )}
          </button>
        ))}
        {admin && <button onClick={addProduct}>+ Produit</button>}
      </div>

      <h3 style={{ marginTop: 30 }}>Panier</h3>
      <ul>
        {cart.length === 0 && <li>Aucun article</li>}
        {cart.map(item => (
          <li key={item.id}>
            {item.qty} x {item.name} — {(item.price * item.qty).toFixed(2)} €
            <button style={{ marginLeft: 8 }} onClick={() => removeFromCart(item.id)}>
              Supprimer
            </button>
          </li>
        ))}
      </ul>

      <div style={{ marginTop: 20 }}>
        <strong>
          Total : {cart.reduce((sum, item) => sum + item.price * item.qty, 0).toFixed(2)} €
        </strong>
      </div>

      <button style={{ marginTop: 20 }} onClick={handlePrint} disabled={cart.length === 0}>
        Imprimer le ticket
      </button>
      <div style={{ minHeight: 24, marginTop: 8 }}>{status}</div>

      {/* Historique des ventes */}
      <h3 style={{ marginTop: 40 }}>Historique des ventes</h3>
      <table style={{ width: "100%", fontSize: "0.95em" }}>
        <thead>
        <tr>
          <th>Date</th>
          <th>Vendeur</th>
          <th>Articles</th>
          <th>Total</th>
        </tr>
        </thead>
        <tbody>
        {sales.length === 0 && <tr>
          <td colSpan={4}>Aucune vente</td>
        </tr>}
        {sales.map(sale => (
          <tr key={sale.id}>
            <td>{sale.date}</td>
            <td>{sale.user}</td>
            <td>
              {sale.cart.map(i => `${i.qty}x${i.name}`).join(", ")}
            </td>
            <td>{sale.total.toFixed(2)} €</td>
          </tr>
        ))}
        </tbody>
      </table>
    </div>
  );
}
