"use strict";
const electron = require("electron");
const path = require("path");
const utils = require("@electron-toolkit/utils");
const lowdb = require("lowdb");
const node = require("lowdb/node");
const icon = path.join(__dirname, "../../resources/icon.png");
const dbFile = path.join(__dirname, "../../db/data.json");
const adapter = new node.JSONFile(dbFile);
const defaultData = { products: [], sales: [] };
const db = new lowdb.Low(adapter, defaultData);
(async () => {
  await db.read();
  db.data ||= { products: [], sales: [] };
  await db.write();
})();
electron.ipcMain.handle("get-products", async () => {
  await db.read();
  return db.data?.products || [];
});
electron.ipcMain.handle("add-product", async (_e, prod) => {
  await db.read();
  const p = { ...prod, id: Date.now().toString() };
  db.data.products.push(p);
  await db.write();
  return p;
});
electron.ipcMain.handle("delete-product", async (_e, id) => {
  await db.read();
  db.data.products = db.data.products.filter((p) => p.id !== id);
  await db.write();
  return true;
});
electron.ipcMain.handle("get-sales", async () => {
  await db.read();
  return db.data?.sales || [];
});
electron.ipcMain.handle("sell-cart", async (_e, items, printerIp) => {
  await db.read();
  let total = 0;
  const products = db.data.products;
  const saleItems = items.map(({ id, qty }) => {
    const p = products.find((x) => x.id === id);
    total += p.price * qty;
    p.stock -= qty;
    return { id, name: p.name, price: p.price, qty };
  });
  const sale = { id: Date.now().toString(), items: saleItems, total, date: (/* @__PURE__ */ new Date()).toISOString() };
  db.data.sales.push(sale);
  await db.write();
  return { ok: true, sale };
});
electron.ipcMain.handle("print-ticket", async (_event, ticketXml, printerUrl) => {
  try {
    const response = await fetch(printerUrl, {
      method: "POST",
      headers: { "Content-Type": "text/xml" },
      body: ticketXml
    });
    const result = await response.text();
    console.log("Impression, retour Epson/mock :", result);
    return { success: true, response: result };
  } catch (err) {
    console.error("Erreur impression Epson:", err);
    return { success: false, error: err.message };
  }
});
function createWindow() {
  const mainWindow = new electron.BrowserWindow({
    width: 900,
    height: 670,
    show: false,
    autoHideMenuBar: true,
    ...process.platform === "linux" ? { icon } : {},
    webPreferences: {
      preload: path.join(__dirname, "../preload/index.js"),
      sandbox: false
    }
  });
  mainWindow.on("ready-to-show", () => {
    mainWindow.show();
  });
  mainWindow.webContents.setWindowOpenHandler((details) => {
    electron.shell.openExternal(details.url);
    return { action: "deny" };
  });
  if (utils.is.dev && process.env["ELECTRON_RENDERER_URL"]) {
    mainWindow.loadURL(process.env["ELECTRON_RENDERER_URL"]);
  } else {
    mainWindow.loadFile(path.join(__dirname, "../renderer/index.html"));
  }
}
electron.app.whenReady().then(() => {
  utils.electronApp.setAppUserModelId("com.electron");
  electron.app.on("browser-window-created", (_, window) => {
    utils.optimizer.watchWindowShortcuts(window);
  });
  electron.ipcMain.on("ping", () => console.log("pong"));
  createWindow();
  electron.app.on("activate", function() {
    if (electron.BrowserWindow.getAllWindows().length === 0) createWindow();
  });
});
electron.app.on("window-all-closed", () => {
  if (process.platform !== "darwin") {
    electron.app.quit();
  }
});
